//
//  AAILivenessConfig.h
//  AAILivenessSDK
//
//  Created by advance on 2022/11/17.
//  Copyright © 2022 Advance.ai. All rights reserved.
//

#import <UIKit/UIKit.h>
#import <AAILivenessSDK/AAIDetectionConstant.h>
#import <AAILivenessSDK/AAIAuditImageConfig.h>
#import <AAILivenessSDK/AAIVideoConfig.h>
#import <AAICore/AAICConstant.h>

NS_ASSUME_NONNULL_BEGIN

__attribute__((visibility("default")))
@interface AAILivenessConfig : NSObject

/// The normal state color of the avatar preview area's border. Default is 0xF5F5F5.
///
/// @since 4.0.0
@property(nonatomic, nullable) UIColor *normalAvatarBorderColor;

/// The hightlight state color of the avatar preview area's border. Default is 0x5BC413.
///
/// @since 4.0.0
@property(nonatomic, nullable) UIColor *highlightAvatarBorderColor;

/// The hightlight color of the inner ellipse animation line of the 3D mode(near/distant mode). Default is 0x5BC413.
///
/// @since 4.0.0
@property(nonatomic, nullable) UIColor *innerAnimatedEllipseLineColor;

/// The color of the ellipse dashed line that appears during the liveness detection. Default is white color.
///
/// @since 4.0.0
@property(nonatomic, nullable) UIColor *innerDashedEllipseLineColor;

/// This value represents the timeout interval during the silent(preparation) stage.
///
/// @warning The range of values should be [10s, 60s], default is 50s.
///
/// @since 4.0.0
@property(nonatomic) NSInteger silentTimeout;

/// This value represents the timeout interval for action detection, the default is 10s.
///
/// @warning The range of values should be [10s, 60s], default is 10s.
///
/// @since 4.0.0
@property(nonatomic) NSInteger actionTimeout;

/// This value actually represents the timeout duration in distant/close mode(3D mode).
/// This time duration does not include network request duration.
///
/// @warning The range of values should be [10s, 60s], default is 50s.
///
/// @since 4.0.0
@property(nonatomic) NSInteger distantNearTimeout;

/// Used to mark the plugin type. Default is `AAILIBPluginTypeStandard`. This value is used for distinguishing different plugins.
///
/// @since 3.4.0
@property(nonatomic) AAICPluginType pluginType;

/// An optional string value used to conveniently query liveness detection results.
/// Call the `/liveness/ext/v1/generate-signature-id` API to generate this value, then assign it to
/// this field before starting a new liveness flow.
///
/// @warning The value must be unique per liveness flow. Passing a previously used value will result in a parameter error in SDK.
///
/// @since 3.6.0
@property(nonatomic, nullable, copy) NSString *signatureId;

/// User binding (strongly recommended).
/// You can use this property to pass your user unique identifier to us,
/// we will establish a mapping relationship based on the identifier.
/// It is helpful for us to check the log when you encountering problems.
///
/// @since 4.0.0
@property(nonatomic, copy, nullable) NSString *userId;

/// A custom string used to associate the current liveness process.
/// @note Note that this field must be paired with ticket initialization method.
///
/// @since 4.0.0
@property(nonatomic, copy, nullable) NSString *queryId;

/// The value required for the ticket initialization method (one of the SDK's three initialization approaches: static key, license, ticket)
/// You must call the `openapi/auth/ticket/v1/create` API to generate this value , then assign it to
/// this field before starting a new liveness flow.
///
/// @warning The value must be unique per liveness flow. Passing a previously used value will result in a parameter error in SDK.
///
/// @since 4.0.0
@property(nonatomic, copy, nullable) NSString *ticket;

/// A configuration object related to audit image collection. 
/// @since 4.0.0
@property(nonatomic, nullable) AAIAuditImageConfig *auditImageConfig;

/// A configuration object related to video recording. You can use it to enable video recording and set the maximum recording duration.
/// 
/// @code
/// // Generate a default video configuration object
/// AAIVideoConfig *videoConfig = [AAIVideoConfig defaultConfig];
/// // enable video recording
/// videoConfig.recordStage = AAIVideoRecordStageAll;
/// // set maximum duration (optional, default is 0, which means no limit)
/// videoConfig.maxRecordDuration = 60;
/// @endcode
///
/// @since 4.0.0
@property(nonatomic, nullable) AAIVideoConfig *videoConfig;

/// Custom model bundle path. If not set, the SDK will use the default model bundle.
/// @warning This property is experimental and may change or be removed in future versions.
/// @since 4.0.0
@property(nonatomic, copy, nullable) NSString *modelBundlePath;

/// Whether to detect face occlusion. The default value is NO.
///
/// @since 4.0.0
@property(nonatomic) BOOL detectOcclusion;

/// The size(width) of the resulting image(`img` of `AAILivenessSuccessResult`). Default result image size is 600x600
///
/// @note
/// This value can be set in the interval [300, 1000], the larger the value, the more memory the SDK will consume.
///
/// @since 4.0.0
@property(nonatomic) CGFloat resultPictureSize;

/// Alias for the liveness verification flow. Passing an invalid value will trigger LIVENESS_TYPE_NOT_FOUND.
///
/// @since 4.0.0
@property(nonatomic, nullable) NSString *livenessType;

- (instancetype)init NS_UNAVAILABLE;

/// @since 4.0.0
+ (AAILivenessConfig *)defaultConfig;

@end

NS_ASSUME_NONNULL_END
