//
//  AAILDSDK.h
//  AAILivenessSDK
//
//  Created by Advance.ai on 2019/2/21.
//  Copyright © 2019 Advance.ai. All rights reserved.
//

#import <UIKit/UIKit.h>
#import <AAILivenessSDK/AAIDetectionConstant.h>
#import <AAILivenessSDK/AAILocalizationUtil.h>
#import <AAILivenessSDK/AAILivenessWrapView.h>
#import <AAILivenessSDK/AAIVideoConfig.h>
#import <AAILivenessSDK/AAIVideoRecordResult.h>
#import <AAILivenessSDK/AAIAdditionalConfig.h>
#import <AAILivenessSDK/AAILDInitConfig.h>

NS_ASSUME_NONNULL_BEGIN

__attribute__((visibility("default")))
@interface AAILivenessSDK : NSObject

/// This method is also used to initialize the SDK.
/// @param config config
///
/// @discussion
/// Currently, there are three ways to initialize the SDK:
///
/// 1. Static Key Initialization: You need to pass accessKey and secretKey through the AAILDStaticKeyModeConfig class.
/// For example:
///
/// @code
/// AAILDStaticKeyModeConfig *config = [[AAILDStaticKeyModeConfig alloc] init];
/// config.accessKey = @"your-accessKey";
/// config.secretKey = @"your-secretKey";
/// config.market = AAILivenessMarketIndonesia; // Set your market
/// config.isGlobalService = NO; // Set whether the SDK is used as a global service
/// [AAILivenessSDK initWithConfig:config];
/// AAILivenessViewController *vc = [[AAILivenessViewController alloc] init];
/// [self.navigationController pushViewController:vc animated:YES];
/// @endcode
///
/// 2. License Initialization: You need to pass the license content through the AAILDLicenseModeConfig class.
/// For example:
///
/// @code
/// AAILDLicenseModeConfig *config = [[AAILDLicenseModeConfig alloc] init];
/// config.market = AAILivenessMarketIndonesia; // Set your market
/// config.isGlobalService = NO; // Set whether the SDK is used as a global service
/// [AAILivenessSDK initWithConfig:config];
/// NSString *result = [AAILivenessSDK configLicenseAndCheck: @"your-license-content"];
/// if ([result isEqualToString:@"SUCCESS"]) {
///     AAILivenessViewController *vc = [[AAILivenessViewController alloc] init];
///     [self.navigationController pushViewController:vc animated:YES];
/// }
/// @endcode
///
/// 3. On-Premise Initialization: You need to pass the server address through the AAILDOnPremiseModeConfig class.
/// For example:
///
/// @code
/// AAILDOnPremiseModeConfig *config = [[AAILDOnPremiseModeConfig alloc] initWithDomain:@"your-server-address" accessKey:@"your-accessKey" secretKey:@"your-secretKey"];
/// [AAILivenessSDK initWithConfig:config];
/// AAILivenessViewController *vc = [[AAILivenessViewController alloc] init];
/// [self.navigationController pushViewController:vc animated:YES];
/// @endcode
///
/// @since 4.1.0
+ (void)initWithConfig:(id<AAILDInitConfig>)config;

+ (NSString *)sdkVersion;

/// This method is used to set the license content.
/// @param license license content
/// @return error infomation. "SUCCESS" means the verification license passed; "LICENSE_EXPIRE" means the license is expired; "APPLICATION_ID_NOT_MATCH"  means your app bundle identifier does not match the license.
///
/// The license content is obtained by your server calling our openapi.
///
/// If the license verification is passed, the SDK will cache the license information in memory, that is, you only need to call this method once, after that you can directly display the liveness page, no need to call this method again.
/// If you call this method every time before the AAILivenessViewController page is displayed, that's fine.
+ (NSString *)configLicenseAndCheck:(NSString *)license;

@end

@interface AAILivenessSDK(DeprecatedOrUnavailable)

/// This method is also used to initialize the SDK, compared to the above method, this method does not need to pass accessKey and secretKey, so as to prevent your accesskey from being leaked.
/// You need to bind the your app bundle id to the accessKey secretKey on our cms website before using the SDK.
/// @param market market
///
/// If you use this initialization method, then you need to call the "configLicenseAndCheck:" method before displaying the AAILivenessViewController page. For example:
/// @code
/// [AAILivenessSDK initWithMarket:AAILivenessMarketIndonesia];
/// NSString *result = [AAILivenessSDK configLicenseAndCheck: @"your-license-content"];
/// if ([result isEqualToString:@"SUCCESS"]) {
///     AAILivenessViewController *vc = [[AAILivenessViewController alloc] init];
///     [self.navigationController pushViewController:vc animated:YES];
/// }
/// @endcode
///
/// @since 1.0.8
/// @deprecated This method is unavailable from 4.1.0
+ (void)initWithMarket:(AAILivenessMarket)market AAI_LD_UNAVAILABLE_FROM(4.1.0, "Using `initWithConfig:` instead");

/// This method is similar to the method "initWithMarket:", the only difference is that this method can specify whether the SDK is used as a global service,
/// @param market market
/// @param isGlobalService whether the SDK is used as a global service. Pass YES indicates that the SDK is used as a global service, If NO is passed, this method is equivalent to the method "initWithMarket:"
///
/// @since 1.2.5
/// @deprecated This method is unavailable from 4.1.0
+ (void)initWithMarket:(AAILivenessMarket)market isGlobalService:(BOOL)isGlobalService AAI_LD_UNAVAILABLE_FROM(4.1.0, "Using `initWithConfig:` instead");

/// This method is used to initialize the SDK and this method is the first method that needs to be called before any other method of the SDK can be used.
/// You need to bind the your app bundle id to the accessKey secretKey on our cms website before using the SDK.
///
/// @param accesskey accessKey
/// @param secretkey secretKey
/// @param market market
///
/// This method is the easiest way to initialize the sdk. For higher security requirements, you should call the "initWithMarket:" method to initialize the SDK.
///
/// @since 1.0.0
/// @deprecated This method is unavailable from 4.1.0
+ (void)initWithAccessKey:(NSString *)accesskey secretKey:(NSString *)secretkey market:(AAILivenessMarket)market AAI_LD_UNAVAILABLE_FROM(4.1.0, "Using `initWithConfig:` instead");

/// This method is similar to the method "initWithAccessKey:secretKey:market:", the only difference is that this method can specify whether the SDK is a global service,
/// @param accesskey accessKey
/// @param secretkey secretKey
/// @param market market
/// @param isGlobalService whether the SDK is used as a global service. Pass YES indicates that the SDK is used as a global service, If NO is passed, this method is equivalent to the method "initWithAccessKey:secretKey:market:"
/// @deprecated This method is unavailable from 4.1.0
+ (void)initWithAccessKey:(NSString *)accesskey secretKey:(NSString *)secretkey market:(AAILivenessMarket)market isGlobalService:(BOOL)isGlobalService AAI_LD_UNAVAILABLE_FROM(4.1.0, "Using `initWithConfig:` instead");

/// @since 1.1.0
+ (void)configQueryId:(NSString *)queryId AAI_LD_UNAVAILABLE_FROM(4.0.0, "Using `queryId` of `AAIAdditionalConfig` instead");

/// @since 1.1.0
+ (void)configTicket:(NSString *)ticket AAI_LD_UNAVAILABLE_FROM(4.0.0, "Using `ticket` of `AAIAdditionalConfig` instead");

/// @since 1.1.0
+ (void)configUserId:(NSString *)userId AAI_LD_UNAVAILABLE_FROM(4.0.0, "Using `userId` of `AAIAdditionalConfig` instead");

/// This method is used to customize the size of the resulting image, which is actually the size of the image obtained in the 'onDetectionComplete:' callback.
/// This method should be called before the AAILivenessViewController is initialized.
/// @param size Image size
///
/// Note that the default result image size is 600x600. This value can be set in the interval [300, 1000], the larger the value, the more memory the SDK will consume.
/// @since 1.1.2
+ (void)configResultPictureSize:(CGFloat)size AAI_LD_UNAVAILABLE_FROM(4.0.0, "Using `resultPictureSize` of `AAIAdditionalConfig` instead");

/// This method is used to set the timeout interval for action detection, the default is 10s.
/// This method should be called before the AAILivenessViewController is initialized.
/// @param actionTimeout timeout in second
/// @deprecated This method is unavailable from v4.0.0.
/// @since 1.1.7
+ (void)configActionTimeoutSeconds:(NSTimeInterval)actionTimeout AAI_LD_UNAVAILABLE_FROM(4.0.0, "Using `timeoutIntervalInSecForAction` of `AAILivenessConfig` instead");

/// Set whether to detect face occlusion.
/// This method should be called before the AAILivenessViewController is initialized.
///
/// If this value is set to YES, SDK will detect  face occlusion  in preparation phase and action detection phase. The default value is NO.
/// @since 1.2.0
+ (void)configDetectOcclusion:(BOOL)detectOcc AAI_LD_UNAVAILABLE_FROM(4.0.0, "Using `detectOcculsion` of `AAILivenessConfig` instead");

/// This method is used to set the path of `AAIModel.bundle`.
///
/// By default the SDK will look for`AAIModel.bundle` from the `[NSBundle mainBundle]`, if you put the `AAIModel.bundle` file somewhere else, then you must call this method to tell the SDK where the bundle file is located, otherwise the SDK will crash.
/// @param modelBundlePath `AAIModel.bundle` file path.
/// @since 1.2.0
+ (void)configModelBundlePath:(NSString *)modelBundlePath AAI_LD_UNAVAILABLE_FROM(4.0.0, "Using `modelBundlePath` of `AAILivenessConfig` instead");

/// Configure the SDK about recording video.
/// @param videoConfig A AAIVideoConfig instance.
/// @since 2.0.0
+ (void)configVideo:(AAIVideoConfig * _Nullable)videoConfig AAI_LD_UNAVAILABLE_FROM(4.0.0, "Using `videoConfig` of `AAILivenessConfig` instead");

/// @since 2.0.0
+ (nullable AAIVideoConfig *)videoConfig AAI_LD_UNAVAILABLE_FROM(4.0.0, "Using `videoConfig` of `AAILivenessConfig` instead");

/// Get the latest recorded video result synchronously. You can only call this method when video recording is enabled.
///
/// Note that this method can only be called in "detectionFailureBlk" and "detectionSuccessBlk" of AAILivenessViewController, and you should
/// always check if video recording is enabled before calling this method.
///
/// @code
/// AAIVideoConfig *originVConfig = [AAILivenessSDK videoConfig];
/// if (originVConfig != nil && originVConfig.recordStage != AAIVideoRecordStageUnspecified) {
///    AAIVideoRecordResult *videoResult = [AAILivenessSDK syncGetLatestVideoRecordResult];
///    NSLog(@"Video result:%@", videoResult.videoPath);
/// }
/// @endcode
///
/// @since 2.0.0
+ (nullable AAIVideoRecordResult *)syncGetLatestVideoRecordResult AAI_LD_UNAVAILABLE_FROM(4.0.0, "Using method `syncGetVideoRecordResult` of `AAILivenessSuccessResult` or `AAILivenessFailureResult` instead");

/// Get additional configuration for the SDK.
+ (AAIAdditionalConfig *)additionalConfig AAI_LD_UNAVAILABLE_FROM(4.0.0, "Using `livenessConfig` of `AAILivenessViewController` instead");

@end
NS_ASSUME_NONNULL_END

