//
//  AAIIQAScanMainView.m
//  AAIGlobalIQASDK
//
//  Created by advance on 2022/1/10.
//

#import "AAIIQAScanMainViewInternal.h"
#import "AAIGlobalIQAInternalConfig.h"
#import "AAIIQALog.h"

@interface AAIIQALabel : UILabel
@property(nonatomic) UIEdgeInsets padding;
@end
@implementation AAIIQALabel

- (instancetype)initWithFrame:(CGRect)frame
{
    self = [super initWithFrame:frame];
    if (self) {
        _padding = UIEdgeInsetsZero;
    }
    return self;
}

- (CGSize)intrinsicContentSize
{
    CGSize sz = [super intrinsicContentSize];
    return CGSizeMake(sz.width + _padding.left + _padding.right, sz.height + _padding.top + _padding.bottom);
}

- (void)drawTextInRect:(CGRect)rect
{
    [super drawTextInRect: UIEdgeInsetsInsetRect(rect, _padding)];
}

@end

static void *FlashShouldVisibleContext = &FlashShouldVisibleContext;
static void *AAIIQAIsFlashOffContext = &AAIIQAIsFlashOffContext;
static void *CurrCameraPositionContext = &CurrCameraPositionContext;
static void *DectionPageSDKStateContext = &DectionPageSDKStateContext;

@implementation AAIIQAScanMainView
@synthesize viewModel = _viewModel;

- (instancetype)initWithFrame:(CGRect)frame
{
    self = [super initWithFrame:frame];
    if (self) {
        _hasAddedObserver = NO;
    }
    return self;
}

- (void)setBackgroundColor:(UIColor *)backgroundColor
{
    [super setBackgroundColor:backgroundColor];
    
    if (backgroundColor) {
        _viewfinderView.maskColor = backgroundColor;
    }
}

- (void)layoutSubviews
{
    [super layoutSubviews];
}

- (void)willTransitionToOrientation:(UIDeviceOrientation)orientation
{
    if (UIDeviceOrientationIsPortrait(orientation)) {
        if (_viewModel.state == AAIIQAStateDetecting) {
            [self startTimer];
        }
        
        [_iqaView willTransitionToOrientation:orientation];
        
        // cameraView layout
        if (_iqaViewLanCons) {
            [NSLayoutConstraint deactivateConstraints: _iqaViewLanCons];
        }
        [NSLayoutConstraint activateConstraints:self.iqaViewPorCons];
        
        // topLabel layout
        [self updateTopStackViewLayout:YES];
        
        [self updateTipViewLayout:YES];
        
        _viewModel.isPortrait = YES;
        
        // Update navBar title
        NSAttributedString *navAttrTitle = [AAIGlobalIQAInternalConfig shareConfig].originConfig.uiConfig.navBarAttrTitle;
        if (!navAttrTitle) {
            _navView.titleLabel.text = _viewModel.navBarTitle;
        } else {
            _navView.titleLabel.attributedText = navAttrTitle;
        }
        
        // Redraw viewfinder
        [_viewfinderView setNeedsDisplay];
    } else if (UIDeviceOrientationIsLandscape(orientation)) {
        if (_viewModel.state == AAIIQAStateDetecting) {
            [self startTimer];
        }
        
        [_iqaView willTransitionToOrientation:orientation];
        
        // cameraView layout
        if (_iqaViewPorCons) {
            [NSLayoutConstraint deactivateConstraints: _iqaViewPorCons];
        }
        [NSLayoutConstraint activateConstraints:self.iqaViewLanCons];
        
        // topLabel layout
        [self updateTopStackViewLayout:NO];
        
        [self updateTipViewLayout:NO];
        
        _viewModel.isPortrait = NO;
        
        // Update navBar title
        NSAttributedString *navAttrTitle = [AAIGlobalIQAInternalConfig shareConfig].originConfig.uiConfig.navBarAttrTitle;
        if (!navAttrTitle) {
            _navView.titleLabel.text = _viewModel.navBarTitle;
        } else {
            _navView.titleLabel.attributedText = navAttrTitle;
        }
        
        // Redraw viewfinder
        [_viewfinderView setNeedsDisplay];
    }
    
}

- (void)loadCameraView
{
    [self addCameraView];
}

- (void)loadAdditionalUI
{
    AAIGlobalIQAConfig *originConfig = [AAIGlobalIQAInternalConfig shareConfig].originConfig;
    AAIGlobalIQAUIConfig *uiConfig = originConfig.uiConfig;
    // 1. Add navigationBar view
    if (uiConfig.navBarVisible) {
        [self addNavBarView];
    }
    
    if (originConfig.operatingMode != AAIIQAOperatingModePhoto) {
        // If not in photo mode, try to display countdown timer label
        // (&& _navView != nil, because timerLabel layout depends on _navView)
        if (uiConfig.timerLabelVisible && _navView != nil) {
            [self addTimeView];
        }
    }
    
    // 3. Add top label
    if (uiConfig.topDescViewVisible) {
        [self addTopLabelView];
    }
    
    // 4. Add tip view
    [self addTipView];
}

- (void)addNavBarView
{
    AAIGlobalIQAUIConfig *uiConfig = [AAIGlobalIQAInternalConfig shareConfig].originConfig.uiConfig;
    
    // NavBar
    AAIIQANavBarView *navView = [[AAIIQANavBarView alloc] init];
    navView.translatesAutoresizingMaskIntoConstraints = NO;
    [self addSubview:navView];
    NSArray *navbarCons = @[
        [navView.topAnchor constraintEqualToAnchor:self.topAnchor],
        [navView.leftAnchor constraintEqualToAnchor:self.leftAnchor],
        [navView.rightAnchor constraintEqualToAnchor:self.rightAnchor],
    ];
    [NSLayoutConstraint activateConstraints:navbarCons];
    _navView = navView;
    
    // Cycle Camera button
    UIImage *cycleCameraImg = [AAIIQACommUtil imgWithName:@"iqa_transform_camera"];
    AAIIQAExtButton *cycleCameraBtn = [AAIIQAExtButton buttonWithType:UIButtonTypeCustom];
    cycleCameraBtn.touchExtSize = CGSizeMake(40, 40);
    [cycleCameraBtn setImage:cycleCameraImg forState:UIControlStateNormal];
    
    cycleCameraBtn.translatesAutoresizingMaskIntoConstraints = NO;
    NSArray *cycleCameraBtnCons = @[
        [cycleCameraBtn.widthAnchor constraintEqualToConstant:cycleCameraImg.size.width],
        [cycleCameraBtn.heightAnchor constraintEqualToConstant:cycleCameraImg.size.height],
    ];
    [NSLayoutConstraint activateConstraints:cycleCameraBtnCons];
    [_navView pushBarItem:cycleCameraBtn marginRight:0];
    
    // Flash button
    UIImage *flashImg = [AAIIQACommUtil imgWithName:@"iqa_light_off"];
    AAIIQAExtButton *flashBtn = [AAIIQAExtButton buttonWithType:UIButtonTypeCustom];
    flashBtn.touchExtSize = CGSizeMake(40, 40);
    [flashBtn setImage:flashImg forState:UIControlStateNormal];
    [flashBtn setImage:[AAIIQACommUtil imgWithName:@"iqa_light_on"] forState:UIControlStateSelected];
    
    flashBtn.translatesAutoresizingMaskIntoConstraints = NO;
    NSArray *flashBtnCons = @[
        [flashBtn.widthAnchor constraintEqualToConstant:flashImg.size.width],
        [flashBtn.heightAnchor constraintEqualToConstant:flashImg.size.height],
    ];
    [NSLayoutConstraint activateConstraints:flashBtnCons];
    [_navView pushBarItem:flashBtn marginRight:0];
    
    flashBtn.hidden = !(uiConfig.lightBtnVisible);
    cycleCameraBtn.hidden = !(uiConfig.flipCameraBtnVisible);
    
    if (!uiConfig.navBarAttrTitle) {
        _navView.titleLabel.textColor = uiConfig.navBarTitleTextColor;
    }
    
    _navView.backgroundColor = uiConfig.navBarBackgroundColor;
    
    _flashBtn = flashBtn;
    [_flashBtn addTarget:self action:@selector(tapFlashBtn) forControlEvents:UIControlEventTouchUpInside];
    _cycleCameraBtn = cycleCameraBtn;
    [_cycleCameraBtn addTarget:self action:@selector(tapCycleCameraBtnBtn) forControlEvents:UIControlEventTouchUpInside];
}

- (void)addCameraView
{
    _iqaView = [[AAIIQACameraWrapperView alloc] init];
    _iqaView.translatesAutoresizingMaskIntoConstraints = NO;
    [self addSubview:_iqaView];
    
    AAIGlobalIQAInternalConfig *config = [AAIGlobalIQAInternalConfig shareConfig];
    BOOL isPortrait = [AAIIQACommUtil isPortraitOrientation];
    AAIIQAOrientation customOrien = config.originConfig.uiConfig.orientation;
    if (customOrien != AAIIQAOrientationAuto) {
        isPortrait = (customOrien == AAIIQAOrientationLandscape) ? NO : YES;
    }
    if (isPortrait) {
        [NSLayoutConstraint activateConstraints:self.iqaViewPorCons];
    } else {
        [NSLayoutConstraint activateConstraints:self.iqaViewLanCons];
    }
    // 3. Add viewfinder view
    _viewfinderView = [[AAIIQAMaskView alloc] init];
    _viewfinderView.backgroundColor = [UIColor clearColor];
    _viewfinderView.maskColor = self.backgroundColor;
    _iqaView.layer.masksToBounds = YES;
    [_iqaView addSubview:_viewfinderView];
    _viewfinderView.translatesAutoresizingMaskIntoConstraints = NO;
    NSArray<NSLayoutConstraint *> *vfCons = @[
        [_viewfinderView.topAnchor constraintEqualToAnchor:_iqaView.topAnchor],
        [_viewfinderView.leftAnchor constraintEqualToAnchor:_iqaView.leftAnchor],
        [_viewfinderView.bottomAnchor constraintEqualToAnchor:_iqaView.bottomAnchor],
        [_viewfinderView.rightAnchor constraintEqualToAnchor:_iqaView.rightAnchor],
    ];
    [NSLayoutConstraint activateConstraints:vfCons];
}

- (NSArray<NSLayoutConstraint *> *)iqaViewPorCons
{
    if (!_iqaViewPorCons) {
        AAIGlobalIQAInternalConfig *config = [AAIGlobalIQAInternalConfig shareConfig];
        
        NSLayoutConstraint *porCons = nil;
        if (self.loadCameraViewPortCons) {
            porCons = self.loadCameraViewPortCons(self, _iqaView);
        }
        if (!porCons) {
            porCons = [_iqaView.centerYAnchor constraintEqualToAnchor:self.centerYAnchor];
        }
        
        _iqaViewPorCons = @[
            [_iqaView.widthAnchor constraintEqualToAnchor:self.widthAnchor multiplier: (1 - 2 * config.finalLRPaddingRatio)],
            [_iqaView.heightAnchor constraintEqualToAnchor:_iqaView.widthAnchor multiplier:0.625],
            [_iqaView.centerXAnchor constraintEqualToAnchor:self.centerXAnchor],
            porCons
        ];
    }
    return _iqaViewPorCons;
}

- (NSArray<NSLayoutConstraint *> *)iqaViewLanCons
{
    if (!_iqaViewLanCons) {
        _iqaViewLanCons = @[
            [_iqaView.heightAnchor constraintEqualToAnchor:self.heightAnchor multiplier:1 constant:-106],
            [_iqaView.widthAnchor constraintEqualToAnchor:_iqaView.heightAnchor multiplier:1.55],
            [_iqaView.centerYAnchor constraintEqualToAnchor:self.centerYAnchor constant:20],
            [_iqaView.centerXAnchor constraintEqualToAnchor:self.centerXAnchor]
        ];
    }
    return _iqaViewLanCons;
}

- (void)addTopLabelView
{
    // top label
    NSString *title = @"";
    AAIIQACardSide cardSide = [AAIGlobalIQAInternalConfig shareConfig].cardSide;
    UIColor *textColor = [AAIGlobalIQAInternalConfig shareConfig].originConfig.uiConfig.primaryTextColor;
    if (cardSide == AAIIQACardSideFront) {
        title = [AAIIQACommUtil localStrForKey:@"iqa_front_side_of_document"];
    } else if (cardSide == AAIIQACardSideBack) {
        title = [AAIIQACommUtil localStrForKey:@"iqa_back_side_of_document"];
    }
    UILabel *titleLabel = [[UILabel alloc] init];
    titleLabel.font = [AAIIQACommUtil fontWithSize:16 isBold:YES];
    titleLabel.text = title;
    titleLabel.textColor = textColor;
    
    UILabel *descLabel = [[UILabel alloc] init];
    descLabel.font = [AAIIQACommUtil fontWithSize:14 isBold:NO];
    descLabel.text = [AAIIQACommUtil localStrForKey:@"iqa_top_desc"];
    descLabel.numberOfLines = 0;
    descLabel.textAlignment = NSTextAlignmentCenter;
    descLabel.textColor = textColor;
    
    UIStackView *topStackView = [[UIStackView alloc] initWithArrangedSubviews:@[titleLabel, descLabel]];
    topStackView.axis = UILayoutConstraintAxisVertical;
    topStackView.alignment = UIStackViewAlignmentCenter;
    topStackView.translatesAutoresizingMaskIntoConstraints = NO;
    topStackView.spacing = 12;
    [self addSubview:topStackView];
    NSArray *topStackViewCons = @[
        [topStackView.leftAnchor constraintEqualToAnchor:_iqaView.leftAnchor constant:10],
        [topStackView.rightAnchor constraintEqualToAnchor:_iqaView.rightAnchor constant:-10],
    ];
    [NSLayoutConstraint activateConstraints:topStackViewCons];
    _topStackView = topStackView;
    _cardSideLabel = titleLabel;
    
    _topDescView = topStackView;
    
    CGFloat lanMarginTop = -12;
    if ([AAIIQACommUtil currScreenSize].height < 667) {
        lanMarginTop = -6;
    }
    _topStackViewBottomPorCons = [_topStackView.bottomAnchor constraintEqualToAnchor:_iqaView.topAnchor constant:-12];
    _topStackViewBottomLanCons = [_topStackView.bottomAnchor constraintEqualToAnchor:_iqaView.topAnchor constant:lanMarginTop];
    
    BOOL isPortrait = [AAIIQACommUtil isPortraitOrientation];
    AAIIQAOrientation customOrien = [AAIGlobalIQAInternalConfig shareConfig].originConfig.uiConfig.orientation;
    if (customOrien != AAIIQAOrientationAuto) {
        isPortrait = (customOrien == AAIIQAOrientationLandscape) ? NO : YES;
    }
    [self updateTopStackViewLayout:isPortrait];
}

- (void)updateTopStackViewLayout:(BOOL)isPortrait
{
    if (isPortrait) {
        _topStackViewBottomLanCons.active = NO;
        _topStackViewBottomPorCons.active = YES;
        
        _cardSideLabel.hidden = NO;
        [_topStackView insertArrangedSubview:_cardSideLabel atIndex:0];
    } else {
        _topStackViewBottomPorCons.active = NO;
        _topStackViewBottomLanCons.active = YES;
        
        [_topStackView removeArrangedSubview:_cardSideLabel];
        _cardSideLabel.hidden = YES;
    }
}

- (void)addTimeView
{
    AAIGlobalIQAUIConfig *uiConfig = [AAIGlobalIQAInternalConfig shareConfig].originConfig.uiConfig;
    AAIIQALabel *label = [[AAIIQALabel alloc] init];
    label.font = [AAIIQACommUtil fontWithSize:14 isBold:NO];
    label.textColor = uiConfig.primaryTextColor;
    label.textAlignment = NSTextAlignmentCenter;
    label.layer.cornerRadius = 12;
    label.layer.borderWidth = 1;
    label.layer.borderColor = label.textColor.CGColor;
    label.padding = UIEdgeInsetsMake(0, 8, 0, 8);
    [self addSubview:label];
    
    label.translatesAutoresizingMaskIntoConstraints = NO;
    NSArray *cons = @[
        [label.topAnchor constraintEqualToAnchor:_navView.bottomAnchor constant:12],
        [AAIIQACommUtil rightContraintOf:label toSV:self constant:-12],
        [label.widthAnchor constraintGreaterThanOrEqualToConstant:40],
        [label.heightAnchor constraintEqualToConstant:24]
    ];
    [NSLayoutConstraint activateConstraints:cons];
    
    _timeLabel = label;
}

- (void)addTipView
{
    AAIGlobalIQAUIConfig *uiConfig = [AAIGlobalIQAInternalConfig shareConfig].originConfig.uiConfig;
    if (!uiConfig.tipViewVisible) return;
    
    BOOL isPortrait = [AAIIQACommUtil isPortraitOrientation];
    AAIIQAOrientation customOrien = uiConfig.orientation;
    if (customOrien != AAIIQAOrientationAuto) {
        isPortrait = (customOrien == AAIIQAOrientationLandscape) ? NO : YES;
    }
    [self updateTipViewLayout:isPortrait];
}

- (void)updateTipViewLayout:(BOOL)isPortrait
{
    AAIGlobalIQAUIConfig *uiConfig = [AAIGlobalIQAInternalConfig shareConfig].originConfig.uiConfig;
    if (!uiConfig.tipViewVisible) return;
    
    // This tip view is unvisible after enter photo mode.
    // so we dont need to update it again.
    if (self.viewModel.state >= AAIIQAStateTakePhoto) {
        return;
    }
    
    if (_tipView == nil) {
        _tipView = [AAIIQATipView tipWithIfPortrait:isPortrait];
        _tipView.translatesAutoresizingMaskIntoConstraints = NO;
        [self addSubview:_tipView];
    }
    
    BOOL preIsPortrait = _tipView.portrait;
    _tipView.portrait = isPortrait;
    
    if (isPortrait) {
        if (!preIsPortrait) {
            // Deactive landscape constranits
            if (_tipViewLanCons != nil) {
                [NSLayoutConstraint deactivateConstraints:_tipViewLanCons];
            }
        }
        
        if (_tipViewPorCons == nil) {
            _tipViewPorCons = @[
                [_tipView.topAnchor constraintEqualToAnchor:_iqaView.bottomAnchor],
                [_tipView.centerXAnchor constraintEqualToAnchor:self.centerXAnchor],
                [_tipView.leftAnchor constraintGreaterThanOrEqualToAnchor:self.leftAnchor],
                [_tipView.rightAnchor constraintLessThanOrEqualToAnchor:self.rightAnchor],
                [_tipView.bottomAnchor constraintEqualToAnchor:self.bottomAnchor],
            ];
            _tipViewPorCons.lastObject.priority = UILayoutPriorityDefaultLow;
        }
        [NSLayoutConstraint activateConstraints:_tipViewPorCons];
    } else {
        // Deactive portrait constranits
        if (preIsPortrait) {
            if (_tipViewPorCons != nil) {
                [NSLayoutConstraint deactivateConstraints:_tipViewPorCons];
            }
        }
        
        if (_tipViewLanCons == nil) {
            _tipViewLanCons = @[
                [_tipView.centerXAnchor constraintEqualToAnchor:self.centerXAnchor],
                [_tipView.bottomAnchor constraintEqualToAnchor:_iqaView.bottomAnchor constant:-10],
            ];
            _tipViewLanCons.lastObject.priority = UILayoutPriorityDefaultLow;
        }
        [NSLayoutConstraint activateConstraints:_tipViewLanCons];
    }
}

- (void)tapFlashBtn
{
    // Update model
    _viewModel.flashButtonOff = _flashBtn.isSelected;
}

- (void)tapCycleCameraBtnBtn
{
    // Update model
    [_viewModel trySwitchCameraPostion];
    
    AVCaptureTorchMode torchMode = [_iqaView flashMode];
    BOOL flashOff = (torchMode == AVCaptureTorchModeOff);
    _viewModel.flashButtonOff = flashOff;
    TLOG(@"_viewModel.flashButtonOff %@, torchMode: %ld", flashOff ? @"YES" : @"NO", torchMode);
}

- (void)bindViewModel:(AAIIQAMainViewModel *)viewModel
{
    _viewModel = viewModel;
    __weak typeof(self) weakSelf = self;
    
    AAIGlobalIQAConfig *originConfig = [AAIGlobalIQAInternalConfig shareConfig].originConfig;
    AAIIQAOperatingMode currOpMode = originConfig.operatingMode;
    if (currOpMode != AAIIQAOperatingModePhoto) {
        // If not in photo mode, start the scanning animation
        [_viewfinderView startAnimation];
    }
    
    BOOL isPor = [AAIIQACommUtil isPortraitOrientation];
    AAIGlobalIQAUIConfig *uiConfig = originConfig.uiConfig;
    AAIIQAOrientation customOrien = uiConfig.orientation;
    if (customOrien != AAIIQAOrientationAuto) {
        isPor = (customOrien == AAIIQAOrientationLandscape) ? NO : YES;
    }
    viewModel.isPortrait = isPor;
    
    // Initialize iqaView and set default camera position
    AVCaptureDevicePosition positon = [viewModel defaultCameraPosition];
    viewModel.currCameraPosition = positon;
    if (positon != AVCaptureDevicePositionUnspecified) {
        _iqaView.currCameraPosition = positon;
    } else {
        // AAIIQAErrorCodeDeviceNotSupport
        if (self.deviceNotSupportBlk) {
            self.deviceNotSupportBlk(self);
        }
    }
    
    if (!uiConfig.navBarAttrTitle) {
        _navView.titleLabel.text = viewModel.navBarTitle;
    } else {
        _navView.titleLabel.attributedText = uiConfig.navBarAttrTitle;
    }
    
    _navView.tapBackBtnBlk = ^(AAIIQANavBarView * _Nonnull navBarView) {
        if (weakSelf.tapNavBackBtnBlk) {
            weakSelf.tapNavBackBtnBlk(weakSelf);
        }
    };
    
    TLOG(@"[AddObserver] %@", viewModel);
    // Add observer
    [viewModel addObserver:self forKeyPath:@"flashButtonOff" options:NSKeyValueObservingOptionNew context:AAIIQAIsFlashOffContext];
    [viewModel addObserver:self forKeyPath:@"currCameraPosition" options:NSKeyValueObservingOptionNew context:CurrCameraPositionContext];
    [viewModel addObserver:self forKeyPath:@"state" options:NSKeyValueObservingOptionNew context:DectionPageSDKStateContext];
    [_iqaView addObserver:self forKeyPath:@"flashAvailable" options:NSKeyValueObservingOptionNew context:FlashShouldVisibleContext];
    _hasAddedObserver = YES;
    
    if (_util == nil) {
        _util = [[AAIIQACommUtil alloc] init];
    }
}

- (void)observeValueForKeyPath:(NSString *)keyPath ofObject:(id)object change:(NSDictionary<NSKeyValueChangeKey,id> *)change context:(void *)context
{
    if (context == AAIIQAIsFlashOffContext) {
        BOOL isOff = [change[NSKeyValueChangeNewKey] boolValue];
        [_iqaView setFlashMode:isOff ? AVCaptureTorchModeOff : AVCaptureTorchModeOn];
        
        _flashBtn.selected = isOff ? NO : YES;
    } else if (context == CurrCameraPositionContext) {
        AVCaptureDevicePosition position = [change[NSKeyValueChangeNewKey] integerValue];
        _iqaView.currCameraPosition = position;
    } else if (context == FlashShouldVisibleContext) {
        BOOL shouldVisible = [change[NSKeyValueChangeNewKey] boolValue];
        TLOG(@"[ScanMainView]Flash button should visible: %d", shouldVisible);
        AAIGlobalIQAUIConfig *uiConfig = [AAIGlobalIQAInternalConfig shareConfig].originConfig.uiConfig;
        if (uiConfig.lightBtnVisible) {
            _flashBtn.hidden = !shouldVisible;
        }
    } else if (context == DectionPageSDKStateContext) {
        AAIIQAState state = [change[NSKeyValueChangeNewKey] integerValue];
        [self sdkStateDidChanged:state];
    }
}

- (void)startRunning
{
    AAIGlobalIQAConfig *originConfig = [AAIGlobalIQAInternalConfig shareConfig].originConfig;
    AAIIQAOperatingMode currOpMode = originConfig.operatingMode;
    
    if (currOpMode == AAIIQAOperatingModePhoto) {
        [self startPhotoMode];
    } else {
        [self startScanMode];
    }
}

- (void)startPhotoMode
{
    // Hide tipView
    _tipView.hidden = YES;
    _timeLabel.hidden = YES;
    
    AAIGlobalIQAUIConfig *config = [AAIGlobalIQAInternalConfig shareConfig].originConfig.uiConfig;
    if (config.flipCameraBtnVisible) {
        _cycleCameraBtn.hidden = NO;
    }
    
    // Directly modify state to enter photo mode
    _viewModel.scanMode = NO;
    _viewModel.state = AAIIQAStateTakePhoto;
    AAIIQACameraWrapperViewPhotoConfig *photoConfig = [[AAIIQACameraWrapperViewPhotoConfig alloc] init];
    photoConfig.region = [AAIGlobalIQAInternalConfig shareConfig].region;
    photoConfig.cardSide = [AAIGlobalIQAInternalConfig shareConfig].cardSide;
    photoConfig.cardType = [AAIGlobalIQAInternalConfig shareConfig].cardType;
    photoConfig.returnEmptyOfOCR = [AAIGlobalIQAInternalConfig shareConfig].returnEmptyOfOCR;
    
    _iqaView.currCameraPosition = _viewModel.currCameraPosition;
    _viewModel.eLogId = [_iqaView startPhotoModeWithConfig:photoConfig delegate:self];
}

- (void)discardCapturedPhoto
{
    [_iqaView discardCapturedPhoto];
}

- (void)startScanMode
{
    _viewModel.scanMode = YES;
    
    _viewModel.state = AAIIQAStateDetecting;
    
    AAIIQACameraWrapperViewScanConfig *scanConfig = [[AAIIQACameraWrapperViewScanConfig alloc] init];
    scanConfig.region = [AAIGlobalIQAInternalConfig shareConfig].region;
    scanConfig.cardSide = [AAIGlobalIQAInternalConfig shareConfig].cardSide;
    scanConfig.cardType = [AAIGlobalIQAInternalConfig shareConfig].cardType;
    scanConfig.detectionTimeoutInterval = [AAIGlobalIQAInternalConfig shareConfig].detectionTimeoutInterval;
    scanConfig.returnEmptyOfOCR = [AAIGlobalIQAInternalConfig shareConfig].returnEmptyOfOCR;
    
    _viewModel.eLogId = [_iqaView startScanModeWithConfig:scanConfig delegate:self];
    
    [self detectionStart];
}

- (void)stopRunning
{
    [_iqaView stopRunning];
    [_viewfinderView stopAnimation];
}

#pragma mark auth delegate
- (void)iqaOnAuthCheckStart
{
    TLOG(@"iqaOnAuthCheckStart");
    if (self.onAuthStart) {
        self.onAuthStart(self);
    }
}

- (void)iqaOnAuthCheckFinish:(NSError * _Nullable)error
{
    TLOG(@"iqaOnAuthCheckFinish");
    if (self.onAuthFinish) {
        self.onAuthFinish(self, error);
    }
}

#pragma mark scan delegate
- (AVCaptureVideoOrientation)iqaPreferredVideoOrientation
{
    if (self.iqaGetPreferredVideoOrientation) {
        return self.iqaGetPreferredVideoOrientation(self);
    }
    return AVCaptureVideoOrientationPortrait;
}

- (void)iqaOnScanModeStart
{
    TLOG(@"iqaOnScanModeStart");
    
    _viewModel.flashButtonVisible = [self.iqaView flashAvailable];
    _viewModel.scanMode = YES;
    
    _viewModel.state = AAIIQAStateDetecting;
    
    [self detectionStart];
}

- (void)iqaOnScanWarnCodeChanged:(AAIIQAWarnCode)warnCode
{
    [self detectionWarn:warnCode];
}

- (void)iqaOnScanSuccess:(UIImage *)rawImage previewImage:(UIImage *)previewImage
{
    TLOG(@"iqaOnScanSuccess");
    [self detectionModeImgDidCaptured:rawImage scaledImg:previewImage];
}

- (void)iqaOnUploadDataStart
{
    TLOG(@"iqaOnUploadDataStart");
    if (self.onUploadStart) {
        self.onUploadStart(self);
    }
}

- (void)iqaOnUploadDataFinish:(NSError *)error
{
    TLOG(@"iqaOnUploadDataFinish");
    if (self.onUploadFinish) {
        self.onUploadFinish(self, error);
    }
}

- (void)iqaOnScanModeComplete:(id<AAIGlobalIQAResultProtocol>)result
{
    TLOG(@"iqaOnScanModeComplete");
    
    [_viewfinderView stopAnimation];
    
    // Stop detection (do not stop the camera)
    _consumeTimes++;
    
    if (result.success) {
        self.onDetectionComplete(result);
    } else {
        NSInteger remainingRetryTimes = [self getRemainingRetryTimes];
        if ([_detectionDelegate respondsToSelector:@selector(iqaOnDetectionTimeout:consumeTimes:remainingRetryTimes:completionHandler:)]) {
            
            void (^blk)(AAIIQADetectionTimeoutDisposition) = ^(AAIIQADetectionTimeoutDisposition disposition) {
                if (disposition == AAIIQADetectionTimeoutDispositionUsePhotoMode) {
                    [self execDispositionAfterTimeout:AAIIQADispositionTypeNext originResult:result];
                } else if (disposition == AAIIQADetectionTimeoutDispositionRetry) {
                    [self retryScanIfAvailable: result];
                }
            };
            
            [_detectionDelegate iqaOnDetectionTimeout:self consumeTimes:_consumeTimes remainingRetryTimes:remainingRetryTimes completionHandler:blk];
        } else {
            [self retryScanIfAvailable:result];
        }
    }
}

- (void)iqaOnScanRemainingSecondsChanged:(NSInteger)remainSeconds totalSeconds:(NSInteger)totalSeconds
{
    TLOG(@"iqaOnScanRemainingSecondsChanged remainSeconds: %ld, totalSeconds: %ld", (long)remainSeconds, (long)totalSeconds);
    [self detectionRemainTimeChanged:remainSeconds totalSeconds:totalSeconds];
}

#pragma mark Photo delegate
- (void)iqaOnPhotoModeStart
{
    TLOG(@"iqaOnPhotoModeStart");
}

- (void)iqaOnCapturePhotoDone:(UIImage *)capturedImage previewImage:(UIImage *)previewImage
{
    TLOG(@"iqaOnCapturePhotoDone");
    if (self.onTakePhotoDone) {
        self.onTakePhotoDone(previewImage, self);
    }
}

- (void)iqaOnPhotoModeComplete:(id<AAIGlobalIQAResultProtocol>)result
{
    TLOG(@"iqaOnPhotoModeComplete");
    // Close page
    self.onDetectionComplete(result);
}

#pragma mark - detection flow
- (void)detectionStart
{
    // callback
    if ([_tipView respondsToSelector:@selector(iqaOnDetectionStart)]) {
        [_tipView iqaOnDetectionStart];
    }
    if ([_detectionDelegate respondsToSelector:@selector(iqaOnDetectionStart)]) {
        [_detectionDelegate iqaOnDetectionStart];
    }
    
    BOOL timerLabelVisible = [AAIGlobalIQAInternalConfig shareConfig].originConfig.uiConfig.timerLabelVisible;
    _timeLabel.hidden = !timerLabelVisible;
    
    // Start scan animation
    [_viewfinderView startAnimation];
}

- (void)detectionRemainTimeChanged:(NSInteger)remainSeconds totalSeconds:(NSInteger)totalSeconds
{
    _timeLabel.text = [NSString stringWithFormat:@"%lu s", (unsigned long)remainSeconds];
    
    NSInteger totalTime = [AAIGlobalIQAInternalConfig shareConfig].detectionTimeoutInterval;
    if ([_tipView respondsToSelector:@selector(iqaOnDetectionRemainTimeChanged:totalTime:contentView:)]) {
        [_tipView iqaOnDetectionRemainTimeChanged:remainSeconds totalTime:totalTime contentView:self];
    }
    if ([_detectionDelegate respondsToSelector:@selector(iqaOnDetectionRemainTimeChanged:totalTime:contentView:)]) {
        [_detectionDelegate iqaOnDetectionRemainTimeChanged:remainSeconds totalTime:totalTime contentView:self];
    }
}

- (NSInteger)getRemainingRetryTimes
{
    NSInteger maxRetryTimes = [AAIGlobalIQAInternalConfig shareConfig].originConfig.maxRetryTimes;
    NSInteger remainingRetryTimes = (maxRetryTimes > 0) ? (maxRetryTimes + 1 - _consumeTimes) : 0;
    return remainingRetryTimes;
}

- (NSInteger)retryTimes
{
    NSInteger retryTimes = _consumeTimes - 1;
    retryTimes = MAX(retryTimes, 0);
    return retryTimes;
}

- (void)retryScanIfAvailable:(id<AAIGlobalIQAResultProtocol>)originResult
{
    NSInteger remainingRetryTimes = [self getRemainingRetryTimes];
    TLOG(@"[remainingRetryTimes %ld]", remainingRetryTimes);
    if (remainingRetryTimes <= 0) {
        // No more retry times
        [self execDispositionAfterTimeout:AAIIQADispositionTypeNext originResult:originResult];
    } else {
        TLOG(@"[Retry scan]");
        [self execDispositionAfterTimeout:AAIIQADispositionTypeRestart originResult:originResult];
    }
}

- (void)execDispositionAfterTimeout:(AAIIQADispositionType)dispositonType
                       originResult:(id<AAIGlobalIQAResultProtocol>)originResult
{
    TLOG(@"execDispositionAfterTimeout");
    switch (dispositonType) {
        case AAIIQADispositionTypeNext: {
            [self detectionTimeout: originResult];
        } break;
            
        case AAIIQADispositionTypeRestart: {
            [self startRunning];
        } break;
        default:
            break;
    }
}

- (void)detectionTimeout:(AAIGlobalIQAResult *)originResult
{
    [_viewfinderView stopAnimation];
    _viewModel.state = AAIIQAStateDetectionTimeout;
    
    // Whether to switch to photo mode and how to switch is handled by onScanTimeout
    if (self.onScanTimeout) {
        self.onScanTimeout(self, [self getRemainingRetryTimes]);
    }
}

- (void)startTimer
{
    BOOL timerLabelVisible = [AAIGlobalIQAInternalConfig shareConfig].originConfig.uiConfig.timerLabelVisible;
    _timeLabel.hidden = !timerLabelVisible;
}

- (void)detectionWarn:(AAIIQAWarnCode)warnCode
{
    if ([_tipView respondsToSelector:@selector(iqaOnDetectionWarn:)]) {
        [_tipView iqaOnDetectionWarn:warnCode];
    }
    if ([_detectionDelegate respondsToSelector:@selector(iqaOnDetectionWarn:)]) {
        [_detectionDelegate iqaOnDetectionWarn:warnCode];
    }
    
    if (warnCode == AAIIQAWarnCodeNoCard) {
        // play audio
        if ([AAIGlobalIQAInternalConfig shareConfig].originConfig.soundPlayEnable) {
            if (![_util isPlaying]) {
                [_util playAudio:@"global_iqa_voice.mp3"];
            }
        }
    }
}

- (void)detectionModeImgDidCaptured:(UIImage *)rawSizeImg scaledImg:(UIImage *)scaledImg
{
    _viewModel.scanMode = YES;
    
    [_iqaView stopRunning];
    [_viewfinderView stopAnimation];
    
    if ([_tipView respondsToSelector:@selector(iqaOnDetectionSucceed:)]) {
        [_tipView iqaOnDetectionSucceed:rawSizeImg];
    }
    if ([_detectionDelegate respondsToSelector:@selector(iqaOnDetectionSucceed:)]) {
        [_detectionDelegate iqaOnDetectionSucceed:rawSizeImg];
    }
    
    _viewModel.state = AAIIQAStateCaptureImgDone;
}

- (void)takePhotoModeImgDidCaptured:(UIImage *)rawSizeImg scaledImg:(UIImage *)scaledImg
{
}

- (void)sdkStateDidChanged:(AAIIQAState)newState
{
    if (newState == AAIIQAStateTakePhoto) {
        // Hide tipView
        _tipView.hidden = YES;
        _timeLabel.hidden = YES;
        
        AAIGlobalIQAUIConfig *config = [AAIGlobalIQAInternalConfig shareConfig].originConfig.uiConfig;
        if (config.lightBtnVisible) {
            _flashBtn.hidden = ![self.iqaView flashAvailable];
        }
        
        if (config.flipCameraBtnVisible) {
            _cycleCameraBtn.hidden = NO;
        }
        
    } else if (newState == AAIIQAStateTakePhotoBegin) {
        // Hide flashBtn cycleCameraBtn
        _flashBtn.hidden = YES;
        _cycleCameraBtn.hidden = YES;
        // Hide tipView
        _tipView.hidden = YES;
    } else if (newState == AAIIQAStateDetecting) {
        
        
    } else if (newState == AAIIQAStateCaptureImgDone) {
        
    } else if (newState == AAIIQAStateDetectionTimeout) {
        
    }
}

- (void)dealloc
{
    if (_hasAddedObserver) {
        [_viewModel removeObserver:self forKeyPath:@"flashButtonOff" context:AAIIQAIsFlashOffContext];
        [_viewModel removeObserver:self forKeyPath:@"currCameraPosition" context:CurrCameraPositionContext];
        [_viewModel removeObserver:self forKeyPath:@"state" context:DectionPageSDKStateContext];
        [_iqaView removeObserver:self forKeyPath:@"flashAvailable" context:FlashShouldVisibleContext];
    }
    TLOG(@"Release AAIIQAScanMainView %p", self);
    [_iqaView stopRunning];
}

@end
