//
//  AAIIQACommUtil.m
//  AAIGlobalIQASDK
//
//  Created by advance on 2022/1/12.
//

#import "AAIIQACommUtil.h"
#import <AVFoundation/AVFoundation.h>
#import <MediaPlayer/MediaPlayer.h>
#import "AAIGlobalIQAInternalConfig.h"
#import "AAIAssets.h"
@interface AAIIQACommUtil()
@property(nonatomic, assign) float volume;
@property(nonatomic, strong) AVAudioPlayer *audioPlayer;
@property(nonatomic, strong) MPVolumeView *volumeView;
@end

@implementation AAIIQACommUtil

static NSString *aaiIqaFinalLan = nil;

+ (NSString *)currLocale
{
    NSLocale *locale = [NSLocale autoupdatingCurrentLocale];
    NSDictionary *components = [NSLocale componentsFromLocaleIdentifier:locale.localeIdentifier];
    NSString *currLan = components[NSLocaleLanguageCode];
    NSString *currContry = components[NSLocaleCountryCode];
    return [NSString stringWithFormat:@"%@_%@", currLan, currContry];
}

+ (nullable NSString *)currLanguageKey
{
    NSArray *array = [NSLocale preferredLanguages];
    if (array.count >= 1) {
        NSString *lanKey = [array objectAtIndex:0];
        NSDictionary *components = [NSLocale componentsFromLocaleIdentifier:lanKey];
        NSString *currLan = components[NSLocaleLanguageCode];
        NSString *currScript = components[NSLocaleScriptCode];
        if (currScript) {
            return [NSString stringWithFormat:@"%@-%@", currLan, currScript];
        } else {
            return currLan;
        }
    }
    
    return nil;
}

+ (NSString *)loadLanFromBundle:(NSBundle *)bundle
{
    /*
     Automatically traverse all lproj files in the folder and try to match the lproj with the current language.
     If there is no match, try to use en.lproj. If there is no en.lproj, use the first lproj.
     If lproj is empty, throw an exception in debug mode.
     */
    NSString *currLan = [self currLanguageKey];
    if (currLan == nil) {
        currLan = @"en";
    }
    
    NSString *expectLproj = [NSString stringWithFormat:@"%@.lproj", currLan];
    NSURL *directoryURL = [[bundle bundleURL] URLByAppendingPathComponent:@"AAIIQALanguageString.bundle"];
    NSFileManager *localFileManager= [[NSFileManager alloc] init];
    NSDirectoryEnumerator *directoryEnumerator =
    [localFileManager enumeratorAtURL:directoryURL
           includingPropertiesForKeys:@[NSURLNameKey]
                              options:NSDirectoryEnumerationSkipsSubdirectoryDescendants | NSDirectoryEnumerationSkipsHiddenFiles
                         errorHandler:nil];
    
    NSMutableArray<NSString *> *fileNames = [NSMutableArray array];
    BOOL existENLproj = NO;
    for (NSURL *fileURL in directoryEnumerator) {
        NSString *tmpName = nil;
        [fileURL getResourceValue:&tmpName forKey:NSURLNameKey error:nil];
        [fileNames addObject:tmpName];
        
        if ([tmpName isEqualToString:@"en.lproj"]) {
            existENLproj = YES;
        }
        
        if ([tmpName isEqualToString:expectLproj]) {
            return currLan;
        }
    }
    
    if (existENLproj) {
        return @"en";
    } else {
        if (fileNames.count > 0) {
            return [fileNames.firstObject stringByReplacingOccurrencesOfString:@".lproj" withString:@""];
        } else {
            NSAssert(NO, @"lproj file not found!");
            return @"";
        }
    }
}

+ (void)resetLanCache
{
    aaiIqaFinalLan = nil;
}

+ (NSString *)currLanForBundle:(NSBundle *)bundle
{
    if (aaiIqaFinalLan == nil) {
        aaiIqaFinalLan = [self loadLanFromBundle:bundle];
    }
    return aaiIqaFinalLan;
}

+ (NSString *)localStrForKey:(NSString *)key
{
    NSBundle *bundle = [[[AAIAssets alloc] init] bundle];
    NSString *lan = [self currLanForBundle:bundle];
    
    AAIGlobalIQAConfig *originConfig = [AAIGlobalIQAInternalConfig shareConfig].originConfig;
    NSString *customLan = originConfig.languageLprojName;
    if (customLan) {
        lan = customLan;
    }
    
    NSString *pathComponent = [NSString stringWithFormat:@"AAIIQALanguageString.bundle/%@.lproj", lan];
    NSString *lprojPath = [bundle.bundlePath stringByAppendingPathComponent:pathComponent];
    NSString *str =  [[NSBundle bundleWithPath:lprojPath] localizedStringForKey:key value:nil table:nil];
    
    AAIGlobalIQAUIConfig *uiconfig = originConfig.uiConfig;
    if (uiconfig.loadLocalizedString) {
        return uiconfig.loadLocalizedString(key, str, lan);
    }
    
    return str;
}

+ (UIImage *)imgWithName:(NSString *)imgName
{
    NSBundle *bundle = [[[AAIAssets alloc] init] bundle];
    UIImage *img = [UIImage imageNamed:imgName inBundle:bundle compatibleWithTraitCollection:nil];
    
    AAIGlobalIQAUIConfig *uiconfig = [AAIGlobalIQAInternalConfig shareConfig].originConfig.uiConfig;
    if (uiconfig.loadImage) {
        return uiconfig.loadImage(imgName, img);
    }
    return img;
}

#pragma mark - play

- (instancetype)init
{
    self = [super init];
    if (self) {
        _volume = 0.5;
    }
    return self;
}

+ (BOOL)isSilent
{
    return [AVAudioSession sharedInstance].outputVolume == 0;
}

- (void)configPlayerVolume:(float)volume
{
    _volume = volume;
    if (_audioPlayer) {
        _audioPlayer.volume = volume;
    }
}

- (void)setVolume:(float)volume
{
    _volume = volume;
    if (_audioPlayer) {
        _audioPlayer.volume = volume;
    }
}

- (BOOL)isPlaying
{
    return _audioPlayer.isPlaying;
}

- (void)playAudio:(NSString *)audioName
{
    if (!audioName) return;
    
    NSBundle *bundle = [[[AAIAssets alloc] init] bundle];
    NSString *lan = [[self class] currLanForBundle:bundle];
    
    NSString *customLan = [AAIGlobalIQAInternalConfig shareConfig].originConfig.languageLprojName;
    if (customLan) {
        lan = customLan;
    }
    
    NSString *pathComponent = [NSString stringWithFormat:@"AAIIQAAudio.bundle/%@.lproj/%@", lan, audioName];
    NSString *path = [bundle.bundlePath stringByAppendingPathComponent:pathComponent];
    NSURL *url = [NSURL URLWithString:path];
    NSError *error = NULL;
    
    AVAudioPlayer *player = _audioPlayer;
    if (player) {
        [player stop];
    }
    
    player = [[AVAudioPlayer alloc] initWithContentsOfURL:url error:&error];
    player.volume = _volume;
    _audioPlayer = player;
    if (!error) {
        [player play];
    }
}

+ (NSLayoutConstraint *)topContraintOf:(UIView *)rawView toSV:(UIView *)sv constant:(CGFloat)constant
{
    if (@available(iOS 11.0, *)) {
        return [rawView.topAnchor constraintEqualToAnchor:sv.safeAreaLayoutGuide.topAnchor constant:constant];
    } else {
        return [rawView.topAnchor constraintEqualToAnchor:sv.topAnchor constant:constant];
    }
}

+ (NSLayoutConstraint *)leftContraintOf:(UIView *)rawView toSV:(UIView *)sv constant:(CGFloat)constant
{
    if (@available(iOS 11.0, *)) {
        return [rawView.leftAnchor constraintEqualToAnchor:sv.safeAreaLayoutGuide.leftAnchor constant:constant];
    } else {
        return [rawView.leftAnchor constraintEqualToAnchor:sv.leftAnchor constant:constant];
    }
}

+ (NSLayoutConstraint *)rightContraintOf:(UIView *)rawView toSV:(UIView *)sv constant:(CGFloat)constant
{
    if (@available(iOS 11.0, *)) {
        return [rawView.rightAnchor constraintEqualToAnchor:sv.safeAreaLayoutGuide.rightAnchor constant:constant];
    } else {
        return [rawView.rightAnchor constraintEqualToAnchor:sv.rightAnchor constant:constant];
    }
}

+ (NSLayoutConstraint *)bottomContraintOf:(UIView *)rawView toSV:(UIView *)sv constant:(CGFloat)constant
{
    if (@available(iOS 11.0, *)) {
        return [rawView.bottomAnchor constraintEqualToAnchor:sv.safeAreaLayoutGuide.bottomAnchor constant:constant];
    } else {
        return [rawView.bottomAnchor constraintEqualToAnchor:sv.bottomAnchor constant:constant];
    }
}

+ (UIColor *)colorWithHexRGB:(NSInteger)hexRGB alpha:(CGFloat)alpha
{
    CGFloat r = ((hexRGB >> 16) & 0xFF) / 255.0;
    CGFloat g = ((hexRGB >> 8) & 0xFF) / 255.0;
    CGFloat b = (hexRGB & 0xFF) / 255.0;
    return [UIColor colorWithRed:r green:g blue:b alpha:alpha];
}

+ (UIFont *)fontWithSize:(CGFloat)size isBold:(BOOL)isBold
{
    UIFont *font = nil;
    NSString *fontName = [AAIGlobalIQAInternalConfig shareConfig].originConfig.uiConfig.fontName;
    if (fontName) {
        font = [UIFont fontWithName:fontName size:size];
    }
    
    if (!font) {
        if (!isBold) {
            font = [UIFont systemFontOfSize:size];
        } else {
            font = [UIFont boldSystemFontOfSize:size];
        }
    }
    return font;
}

+ (BOOL)isPortraitOrientation
{
    BOOL isPortrait = YES;
    UIDeviceOrientation orientation = [UIDevice currentDevice].orientation;
    if (orientation == UIDeviceOrientationUnknown) {
        UIInterfaceOrientation iOrientation = [[UIApplication sharedApplication] statusBarOrientation];
        if (UIInterfaceOrientationIsLandscape(iOrientation)) {
            isPortrait = NO;
        }
    } else if (UIDeviceOrientationIsLandscape(orientation)) {
        isPortrait = NO;
    } else if (UIDeviceOrientationIsPortrait(orientation)) {
        isPortrait = YES;
    } else {
        UIInterfaceOrientation iOrientation = [[UIApplication sharedApplication] statusBarOrientation];
        if (UIInterfaceOrientationIsLandscape(iOrientation)) {
            isPortrait = NO;
        } else {
            isPortrait = YES;
        }
    }
    
    return isPortrait;
}

+ (NSArray *)avaliableCameraPositions
{
    // Check if the camera is available
    if ([UIImagePickerController isSourceTypeAvailable:UIImagePickerControllerSourceTypeCamera] == NO) {
        return nil;
    }
    
    NSMutableArray *array = [[NSMutableArray alloc] init];
    
    if ([UIImagePickerController isCameraDeviceAvailable:UIImagePickerControllerCameraDeviceRear]) {
        [array addObject:@(AVCaptureDevicePositionBack)];
    }
    if ([UIImagePickerController isCameraDeviceAvailable:UIImagePickerControllerCameraDeviceFront]) {
        [array addObject:@(AVCaptureDevicePositionFront)];
    }
    
    if (array.count == 0) {
        return nil;
    }
    return array;
}

#pragma mark - UIImage

+ (UIImage *)imgWithColor:(UIColor *)color size:(CGSize)size
{
    CGRect rect = CGRectMake(0, 0, size.width, size.height);
    UIGraphicsBeginImageContextWithOptions(size, false, [UIScreen mainScreen].scale);
    CGContextRef context = UIGraphicsGetCurrentContext();
    CGContextSetFillColorWithColor(context, color.CGColor);
    CGContextFillRect(context, rect);
    UIImage *img = UIGraphicsGetImageFromCurrentImageContext();
    UIGraphicsEndImageContext();
    return img;
}

#pragma mark - Button

+ (UIButton *)buttonWithTitle:(NSString *)title
{
    UIColor *themeColor = [AAIGlobalIQAInternalConfig shareConfig].originConfig.uiConfig.toolButtonThemeColor;
    UIButton *btn = [UIButton buttonWithType:UIButtonTypeCustom];
    btn.layer.cornerRadius = 6;
    btn.layer.masksToBounds = true;
    btn.titleLabel.font = [self fontWithSize:15 isBold:NO];
    [btn setTitle:title forState:UIControlStateNormal];
    UIImage *normalImg = [self imgWithColor:themeColor size:CGSizeMake(80, 42)];
    UIImage *hightlightedImg = [self imgWithColor:[themeColor colorWithAlphaComponent:0.6] size:CGSizeMake(80, 42)];
    [btn setBackgroundImage:normalImg forState:UIControlStateNormal];
    [btn setBackgroundImage:hightlightedImg forState:UIControlStateHighlighted];
    [btn setTitleColor:UIColor.whiteColor forState:UIControlStateNormal];
    [btn setTitleColor:[UIColor.whiteColor colorWithAlphaComponent:0.5] forState:UIControlStateHighlighted];
    
    return btn;
}

+ (UIButton *)borderButtonWithTitle:(NSString *)title
{
    UIColor *themeColor = [AAIGlobalIQAInternalConfig shareConfig].originConfig.uiConfig.toolButtonThemeColor;
    UIButton *btn = [UIButton buttonWithType:UIButtonTypeCustom];
    btn.titleLabel.font = [self fontWithSize:15 isBold:NO];
    [btn setTitle:title forState:UIControlStateNormal];
    btn.layer.cornerRadius = 6;
    btn.layer.borderWidth = 1;
    btn.layer.masksToBounds = true;
    btn.layer.borderColor = themeColor.CGColor;
    
    [btn setTitleColor:themeColor forState:UIControlStateNormal];
    [btn setTitleColor:[themeColor colorWithAlphaComponent:0.5] forState:UIControlStateHighlighted];
    
    return btn;
}

+ (UIButton *)porButtonWithTitle:(NSString *)title img:(UIImage *)img
{
    UIColor *themeColor = [AAIGlobalIQAInternalConfig shareConfig].originConfig.uiConfig.toolButtonThemeColor;
    UIButton *btn = [UIButton buttonWithType:UIButtonTypeCustom];
    btn.titleLabel.font = [self fontWithSize:14 isBold:NO];
    [btn setTitleColor:themeColor forState:UIControlStateNormal];
    [btn setTitleColor:[themeColor colorWithAlphaComponent:0.5] forState:UIControlStateHighlighted];
    [btn setImage:img forState:UIControlStateNormal];
    [btn setTitle:title forState:UIControlStateNormal];
    [btn setTitleEdgeInsets:UIEdgeInsetsMake(btn.imageView.frame.size.height + 5 ,-btn.imageView.frame.size.width, 0.0,0.0)];
    
    CGSize labelSize = [btn.titleLabel sizeThatFits:UILayoutFittingCompressedSize];
    CGSize imgSize = img.size;
    CGFloat btnWidth = MAX(labelSize.width, imgSize.width);
    CGFloat padding = 0;
    btn.titleEdgeInsets = UIEdgeInsetsMake(imgSize.height + padding, -btnWidth, 0, 0);
    btn.imageEdgeInsets = UIEdgeInsetsMake(-labelSize.height, 0, 0, 0);
    
    btn.translatesAutoresizingMaskIntoConstraints = NO;
    NSArray<NSLayoutConstraint *> *cons = @[
        [btn.widthAnchor constraintEqualToConstant:btnWidth],
        [btn.heightAnchor constraintEqualToConstant:(labelSize.height + imgSize.height + padding)]
    ];
    [NSLayoutConstraint activateConstraints:cons];
    
    return btn;
}

+ (CGSize)currScreenSize
{
    CGSize screenSize = [UIScreen mainScreen].bounds.size;
    
    CGFloat lenMin = 0;
    CGFloat lenMax = 0;
    if (screenSize.width > screenSize.height) {
        lenMin = screenSize.height;
        lenMax = screenSize.width;
    } else {
        lenMin = screenSize.width;
        lenMax = screenSize.height;
    }
    return CGSizeMake(lenMin, lenMax);
}

@end
