//
//  AAIIQAScanController.m
//  AAIGlobalIQASDK
//
//  Created by advance on 2022/1/4.
//

#import "AAIIQAScanController.h"
#import "AAIGlobalIQAInternalConfig.h"
#import "AAIIQAScanMainViewInternal.h"
#import "AAIIQATakePhotoMainViewInternal.h"
#import "AAIIQAMainViewModel.h"
#import "AAIIQACommUtil.h"
#import "AAIIQAHUD.h"
#import "AAIIQAPreviewImgController.h"

@interface AAIIQAScanController ()
{
    CAShapeLayer *_rectMaskLayer;
}
@property(nonatomic) AAIIQAMainViewModel *mainViewModel;
@end

@implementation AAIIQAScanController

- (void)loadView
{
    AAIIQAOperatingMode opMode = [AAIGlobalIQAInternalConfig shareConfig].originConfig.operatingMode;
    
    if (opMode == AAIIQAOperatingModeScanning) {
        // If it is scanning mode only, use AAIIQAScanMainView
        self.view = [[AAIIQAScanMainView alloc] init];
    } else {
        self.view = [[AAIIQATakePhotoMainView alloc] init];
    }
}

- (void)viewDidLoad
{
    [super viewDidLoad];
    
    AAIGlobalIQAConfig *config = [AAIGlobalIQAInternalConfig shareConfig].originConfig;
    AAIIQAOperatingMode opMode = config.operatingMode;
    
    __weak typeof(self) weakSelf = self;
    
    AAIIQAScanMainView *mainView = (AAIIQAScanMainView *)self.view;
    mainView.loadCameraViewPortCons = ^NSLayoutConstraint * _Nonnull(AAIIQAScanMainView * _Nonnull mainView, UIView * _Nonnull cameraView) {
        __strong AAIIQAScanController *strongSelf = weakSelf;
        if (!strongSelf) {
            return nil;
        }
        return [strongSelf loadCameraViewPortCons: cameraView];
    };
    // 2. Load mainView additional UI
    [self loadSDKUI];
    
    _mainViewModel = [[AAIIQAMainViewModel alloc] init];
    if (_mainViewModel.currCameraPosition == AVCaptureDevicePositionUnspecified) {
        [self callbackWithResult:[_mainViewModel resultWithErrorCode:AAIIQAErrorCodeDeviceNotSupport]];
#if AAI_IQA_EXIT_WHEN_DEVICE_NOT_SUPPORT
        return;
#endif
    }
    
    mainView.tapNavBackBtnBlk = ^(AAIIQAScanMainView * _Nonnull originMainView) {
        __strong AAIIQAScanController *strongSelf = weakSelf;
        if (!strongSelf) {
            return;
        }
        [strongSelf closePage:YES];
    };
    
    mainView.onAuthStart = ^(AAIIQAScanMainView * _Nonnull rawView) {
        UIViewController *currVC = weakSelf.navigationController.viewControllers.lastObject;
        [weakSelf beginRequest:currVC];
    };
    
    mainView.onAuthFinish = ^(AAIIQAScanMainView * _Nonnull rawView, NSError * _Nullable error) {
        UIViewController *currVC = weakSelf.navigationController.viewControllers.lastObject;
        [weakSelf endRequest:currVC];
    };
    
    mainView.onUploadStart = ^(AAIIQAScanMainView * _Nonnull rawView) {
        UIViewController *currVC = weakSelf.navigationController.viewControllers.lastObject;
        [weakSelf beginRequest:currVC];
    };
    
    mainView.onUploadFinish = ^(AAIIQAScanMainView * _Nonnull rawView, NSError * _Nullable error) {
        UIViewController *currVC = weakSelf.navigationController.viewControllers.lastObject;
        [weakSelf endRequest:currVC];
    };
    
    mainView.onDetectionComplete = ^(id<AAIGlobalIQAResultProtocol> result) {
        
        if (result.errorCodeEnumValue == AAIIQAErrorCodeScanTimeout) {
            [weakSelf processScanTimeout:result];
        } else {
            [weakSelf callbackWithResult:result];
        }
    };
    
    mainView.onScanTimeout = ^(AAIIQAScanMainView * _Nonnull rawView, NSInteger remainingRetryTimes) {
        __strong AAIIQAScanController *strongSelf = weakSelf;
        if (!strongSelf) {
            return;
        }
        
        if (opMode == AAIIQAOperatingModeScanning) {
            if (remainingRetryTimes <= 0) {
                // Over retry times, directly end sdk
                [weakSelf callbackWithResult:[weakSelf.mainViewModel resultWithErrorCode:AAIIQAErrorCodeScanTimeout]];
            } else {
                // Still have retry times, do nothing here
            }
        }
        
    };
    
    mainView.onTakePhotoDone = ^(UIImage * _Nonnull previewImg, AAIIQAScanMainView * _Nonnull rawView) {
        [weakSelf showPreviewVC:previewImg];
    };
    
    mainView.iqaGetPreferredVideoOrientation = ^AVCaptureVideoOrientation(AAIIQAScanMainView * _Nonnull rawView) {
        __strong AAIIQAScanController *strongSelf = weakSelf;
        if (!strongSelf) {
            return AVCaptureVideoOrientationPortrait;
        }
        
        if ([strongSelf respondsToSelector:@selector(preferredInterfaceOrientationForPresentation)]) {
            return (AVCaptureVideoOrientation)[strongSelf preferredInterfaceOrientationForPresentation];
        }
        
        return AVCaptureVideoOrientationPortrait;
    };
    
    [mainView bindViewModel:self.mainViewModel];
    [mainView startRunning];
}

- (void)processScanTimeout:(id<AAIGlobalIQAResultProtocol>)result
{
}

- (void)viewWillTransitionToSize:(CGSize)size withTransitionCoordinator:(id<UIViewControllerTransitionCoordinator>)coordinator
{
    UIDeviceOrientation deviceOrientation = [UIDevice currentDevice].orientation;
    [(AAIIQAScanMainView *)self.view willTransitionToOrientation:deviceOrientation];
    
    [super viewWillTransitionToSize:size withTransitionCoordinator:coordinator];
}

- (void)callbackWithResult:(id<AAIGlobalIQAResultProtocol>)result
{
    [self callbackWithResult:result animated:YES];
}

- (void)callbackWithResult:(id<AAIGlobalIQAResultProtocol>)result animated:(BOOL)animated
{
    dispatch_async(dispatch_get_main_queue(), ^{
        BOOL callbackAfter = [AAIGlobalIQAInternalConfig shareConfig].originConfig.callbackAfterPageDismissed;
        void (^blk)(void) = ^{
            // callback
            id<AAIGlobalIQADelegate> delegate = [AAIGlobalIQAInternalConfig shareConfig].originConfig.delegate;
            if ([delegate respondsToSelector:@selector(iqaOnDetectionComplete:)]) {
                [delegate iqaOnDetectionComplete:result];
            }
            
            if ([AAIGlobalIQAInternalConfig shareConfig].completionBlk != nil) {
                [AAIGlobalIQAInternalConfig shareConfig].completionBlk(result);
            }
        };
        
        if (!callbackAfter) {
            blk();
            blk = nil;
        }
        
        if (result.errorCodeEnumValue == AAIIQAErrorCodeDeviceNotSupport) {
#if AAI_IQA_EXIT_WHEN_DEVICE_NOT_SUPPORT
            [self.navigationController.presentingViewController dismissViewControllerAnimated:animated completion:blk];
#endif
        } else {
            [self.navigationController.presentingViewController dismissViewControllerAnimated:animated completion:blk];
        }
    });
}


- (void)showPreviewVC:(UIImage *)previewImg
{
    Class previewImgClz = [[AAIGlobalIQAInternalConfig shareConfig].originConfig classOfModuleType:AAIIQAModuleTypePreviewImgController];
    
    AAIIQAPreviewImgController *ppVC = [[previewImgClz alloc] init];
    ppVC.previewImage = previewImg;
    ppVC.title = _mainViewModel.navBarTitle;
    
    ppVC.tapNavBackBtnBlk = ^(AAIIQAPreviewImgController * _Nonnull rawVC) {
        [self callbackWithResult:[self.mainViewModel resultWithErrorCode:AAIIQAErrorCodeUserGiveUp]];
    };
    
    ppVC.tapRetakeBtnBlk = ^(AAIIQAPreviewImgController * _Nonnull rawVC) {
        AAIIQATakePhotoMainView *mainView = (AAIIQATakePhotoMainView *)self.view;
        [mainView discardCapturedPhoto];
        self.mainViewModel.state = AAIIQAStateTakePhoto;
        
        [rawVC.navigationController popViewControllerAnimated:NO];
    };
    
    ppVC.tapContinueBtnBlk = ^(AAIIQAPreviewImgController * _Nonnull rawVC) {
        AAIIQATakePhotoMainView *mainView = (AAIIQATakePhotoMainView *)self.view;
        [mainView tapContinueBtnAction];
    };
    
    [self.navigationController pushViewController:ppVC animated:NO];
}

- (void)dealloc
{
    // If the SDK page is presented muti-time at the same time, then self.view is nil in dealloc method,
    // which will cause [iqaView timeCostInfo]; return nil, then
    // [detail setObject:info[@"avgFrameCost"] forKey:@"frame_avg_cost"]
    // will crash "[__NSDictionaryM setObject:forKey:]: object cannot be nil (key: frame_avg_cost)'"
    if (self.view == nil) return;
    
    UIView *iqaView = [(AAIIQAScanMainView *)self.view iqaView];
    if (iqaView == nil) return;
    
    AAIIQAScanMainView *mainView = (AAIIQAScanMainView *)self.view;
    [mainView stopRunning];
}

# pragma mark - Public API
- (void)setFlashButtonOff:(BOOL)flashButtonOff
{
    if (_mainViewModel) {
        _mainViewModel.flashButtonOff = flashButtonOff;
    }
}

- (BOOL)flashButtonOff
{
    if (!_mainViewModel) return NO;
    return _mainViewModel.flashButtonOff;
}

- (void)closePage:(BOOL)animated
{
    [self callbackWithResult:[self.mainViewModel resultWithErrorCode:AAIIQAErrorCodeUserGiveUp] animated:animated];
}

- (NSLayoutConstraint * _Nullable)loadCameraViewPortCons:(UIView *)cameraView
{
    return nil;
}

- (void)loadSDKUI
{
    [self loadCameraView];
    [self loadAdditionalUI];
}

- (void)loadCameraView
{
    AAIIQAScanMainView *mainView = (AAIIQAScanMainView *)self.view;
    [mainView loadCameraView];
    _cameraView = mainView.iqaView;
}

- (void)loadAdditionalUI
{
    AAIIQAScanMainView *mainView = (AAIIQAScanMainView *)self.view;
    [mainView loadAdditionalUI];
    _navBarView = mainView.navView;
}

- (BOOL)flashOff
{
    if (!_mainViewModel) {
        return YES;
    } else {
        return _mainViewModel.flashButtonOff;
    }
}

- (void)setFlashOff:(BOOL)flashOff
{
    if (_mainViewModel) {
        _mainViewModel.flashButtonOff = flashOff;
    }
}

- (void)beginRequest:(UIViewController * _Nullable)currVC
{
    // Upload image
    [AAIIQAHUD showWaitWithMsg:[AAIIQACommUtil localStrForKey:@"iqa_auth_check"] onView:currVC.view];
}

- (void)endRequest:(UIViewController * _Nullable)currVC
{
    [AAIIQAHUD dismissHUDOnView:currVC.view afterDelay:0];
}

@end
