//
//  AAIGlobalIQAConfig.h
//  AAIGlobalIQASDK
//
//  Created by advance on 2022/1/4.
//

#import <UIKit/UIKit.h>
#import <AAIGlobalIQASDK/AAIIQAConstant.h>

#if __has_include(<AAIGlobalIQAUI/AAIGlobalIQADelegate.h>)
#import <AAIGlobalIQAUI/AAIGlobalIQADelegate.h>
#else
#import "AAIGlobalIQADelegate.h"
#endif

#if __has_include(<AAIGlobalIQAUI/AAIIQADelegate.h>)
#import <AAIGlobalIQAUI/AAIIQADelegate.h>
#else
#import "AAIIQADelegate.h"
#endif

NS_ASSUME_NONNULL_BEGIN

typedef NS_ENUM(NSInteger, AAIIQAModuleType) {
    /// Refers to the `AAIIQAScanController`
    AAIIQAModuleTypeScanController = 1,
    
    /// Refers to the `AAIIQAPreviewImgController`
    AAIIQAModuleTypePreviewImgController = 2,
    
    /// Refer to the `AAIIQAOverlayView` in `AAIIQAScanController`
    AAIIQAModuleTypeOverlayViewOfScanVC = 3,
    
    /// Refer to the `AAIIQAOverlayView` in `AAIIQAPreviewImgController`
    AAIIQAModuleTypeOverlayViewOfPreviewImgVC = 4,
};

typedef NS_ENUM(NSInteger, AAIIQAOperatingMode) {
    /// Default mode(Scanning + Photo). The SDK will first enter the real-time scanning mode, and if it times out, the SDK will automatically switch to the photo mode.
    AAIIQAOperatingModeDefault = 1,
    
    /// Real-time scanning mode. In this mode, the SDK will scan the document in real-time until it times out.
    AAIIQAOperatingModeScanning = 2,
    
    /// Photo mode. In this mode, the SDK can only take photos.
    AAIIQAOperatingModePhoto = 3,
};

/**
 This class is used to configure the SDK appearance.
 */
__attribute__((visibility("default")))
@interface AAIGlobalIQAUIConfig : NSObject

/// The font used by the SDK, if you don't specify it,  the SDK will use the system font.
@property(nonatomic, nullable, copy) NSString *fontName;

/// Page background color. Default is 0xF6F6F5.
@property(nonatomic) UIColor *pageBackgroundColor;

/// Primary text color. Default is 0x333333.
@property(nonatomic) UIColor *primaryTextColor;

/// Viewfinder(Camera overlay view) color. Default is 0x333333.
@property(nonatomic) UIColor *viewfinderColor;

/// Whether to show the flip camera button. Default is YES.
@property(nonatomic) BOOL flipCameraBtnVisible;

/// Whether to show the light button. Default is YES.
@property(nonatomic) BOOL lightBtnVisible;

/// Whether to display the description view above the camera view when the device is in portrait orientation.  Default is YES.
@property(nonatomic) BOOL topDescViewVisible;

/// Whether to show the icon at the bottom when the device is in portrait orientation. Default is YES.
@property(nonatomic) BOOL tipIconVisible;

/// Whether to show the tipView under the camera view. Default is YES.
@property(nonatomic) BOOL tipViewVisible;

/// Whether to show the timer label in the upper right corner of the view. Default is YES.
@property(nonatomic) BOOL timerLabelVisible;


@property(nonatomic) BOOL navBarVisible;

/// Navigation bar title text color. Default is 0x333333.
@property(nonatomic) UIColor *navBarTitleTextColor;

/// Navigation bar background color. Default is clear color.
@property(nonatomic) UIColor *navBarBackgroundColor;

/// The title of the navigation bar. If you specify nil, then the SDK will use the default value.
@property(nonatomic, copy, nullable) NSAttributedString *navBarAttrTitle;


/// Default is UIStatusBarStyleDefault.
@property(nonatomic) UIStatusBarStyle statusBarStyle;

/// Submit & Retake button theme color. Default is 0x30B043 and alpha is 0.5.
@property(nonatomic) UIColor *toolButtonThemeColor;


@property(nonatomic) BOOL takePhotoTipViewVisible;
@property(nonatomic) BOOL takePhotoButtonVisible;

/// The bottom tip view text color in take photo mode. Default is 0x333333.
@property(nonatomic) UIColor *takePhotoTipTextColor;

/// The bottom tip view background color in take photo mode. Default is 0xE0E0E0 and alpha is 0.8.
@property(nonatomic) UIColor *takePhotoTipBackgroundColor;

/// Whether to show the takePhotoTipView's left icon. Default is YES.
@property(nonatomic) BOOL takePhotoTipLeftIconVisible;

@property(nonatomic, copy, nullable) NSAttributedString *takePhotoTipAttrText;


/// The bottom tip view text color in preview photo mode. Default is 0x333333.
@property(nonatomic) UIColor *previewPhotoTipTextColor;

/// The bottom tip view background color in preview photo mode. Default is 0xE0E0E0 and alpha is 0.8.
@property(nonatomic) UIColor *previewPhotoTipBackgroundColor;

/// Whether to show the previewPhotoTipView's left icon. Default is YES.
@property(nonatomic) BOOL previewPhotoTipLeftIconVisible;

@property(nonatomic, copy, nullable) NSAttributedString *previewPhotoTipAttrText;


/// Interface orientation when the SDK is displayed.  Default is AAIIQAOrientationAuto.
@property(nonatomic) AAIIQAOrientation orientation;

/// The left and right margin on x-axis of cameraView when device in portrait mode. Default is nil.
@property(nonatomic, nullable) NSNumber *cameraViewMarginLRInPortraitMode;

/// You can implement this block to customize the images used in the SDK.
///
/// e.g.
/// @code
/// config.uiConfig.loadImage = ^UIImage * _Nonnull(NSString * _Nonnull imgName, UIImage * _Nonnull img) {
///     if ([imgName isEqualToString:@"iqa_scan_successfully"]) {
///         // load custom image
///         UIImage *myImg = [UIImage imageNamed:@"my-custom-img"];
///         return myImg;
///     } else {
///         return img;
///     }
/// };
///
@property(nonatomic, copy, nullable) UIImage * (^loadImage)(NSString *imgName, UIImage *img);

/// You can implement this block to customize the localized strings used in the SDK.
///
/// e.g.
/// @code
/// config.uiConfig.loadLocalizedString = ^NSString * _Nonnull(NSString * _Nonnull key, NSString * _Nonnull value, NSString * _Nonnull language) {
///  if ([key isEqualToString:@"iqa_top_desc"]) {
///     if ([language isEqual:@"en"]) {
///         return @"My test string";
///     }
///  }
///
///  return value;
/// };
/// @endcode
/// For all available keys, see "AAIGlobalIQASDK.framework/Assets.bundle/AAIIQALanguageString.bundle/en.lproj/Localizable.strings".
@property(nonatomic, copy, nullable) NSString * (^loadLocalizedString)(NSString *key, NSString *value, NSString *language);

@end

/**
 This class defines behavior and policies for SDK.
 */
__attribute__((visibility("default")))
@interface AAIGlobalIQAConfig : NSObject

/// Regional ISO code, you can use 2-digit or 3-digit shortcode, see Wikipedia https://zh.wikipedia.org/wiki/ISO_3166-1
@property(nonatomic, copy, readonly) NSString *region;

@property(nonatomic, readonly) AAIIQACardType cardType;
@property(nonatomic, readonly) AAIIQACardSide cardSide;

@property(nonatomic, weak) id<AAIGlobalIQADelegate> delegate;
@property(nonatomic, readonly) AAIGlobalIQAUIConfig *uiConfig;

/// If the ui configuration available for the tipView in scan mode does not meet your needs, you can create a tip view by specifying this property.
@property(nonatomic, weak, nullable) id<AAIIQATipViewDataSource> tipViewDataSource;

/// If the ui configuration available for the tipView in photo mode does not meet your needs, you can create a tip view by specifying this property.
@property(nonatomic, weak, nullable) id<AAIIQATakePhotoTipViewDataSource> takePhotoTipViewDataSource;

/// Detection timeout interval in seconds. Default is 20s.
@property(nonatomic) NSUInteger detectionTimeoutInterval;

/// The duration of the alert view displayed when SDK are about to enter the photo mode,
/// the default is 0s, which means the alert view will not be displayed.
@property(nonatomic) NSUInteger takePhotoAlertViewTimeoutInterval;

/// The maximum times of retries for scanning. Default is 0.
@property(nonatomic) NSInteger maxRetryTimes;

/// Whether to play a prompt audio during the detection process. Default is YES.
@property(nonatomic) BOOL soundPlayEnable;

/// Whether to return null value on fields that recognized as empty, defaults is NO.
@property(nonatomic) BOOL returnEmptyOfOCR;

/// Whether to callback the AAIGlobalIQADelegate method `iqaOnDetectionComplete:` after the view controller is dismissed.
/// Default is NO, which means that the `iqaOnDetectionComplete:` method will be called before the IQA page is closed,
/// If you set YES, then the method `iqaOnDetectionComplete:` is called after the IQA page is dismissed.
@property(nonatomic) BOOL callbackAfterPageDismissed;

/// Specifies the language lproj file to be loaded by the SDK.  e.g. "en" or "id" or "zh-Hans" or "th".
@property(nonatomic, copy, nullable) NSString *languageLprojName;

/// The SDK can have different operating modes, such as real-time scanning mode, photo mode, default mode(Scanning + Photo). Default is 'AAIIQAOperatingModeDefault'.
@property(nonatomic) AAIIQAOperatingMode operatingMode;

/// A configuration object that defines behavior and policies for SDK.
/// @param region Regional ISO code, you can use 2-digit or 3-digit shortcode, see Wikipedia https://zh.wikipedia.org/wiki/ISO_3166-1
/// @param cardType Card type
/// @param cardSide Card side
+ (instancetype)initWithRegion:(NSString *)region cardType:(AAIIQACardType)cardType cardSide:(AAIIQACardSide)cardSide;

/// A default configuration object that defines behavior and policies for SDK. When you using `AAIIQACameraWrapperView`, you can using this method to
/// create a default configuration object  then pass it to `initWithConfig:` method
+ (instancetype)defaultConfig;

/// You can call this method to pass in a custom class and then the SDK will use the class you passed in when it starts.
///
/// The class you pass in must inherit from a specific class, for example, if you customize the scan page, the custom class must inherit from `AAIIQAScanController` and the moduleType must pass `AAIIQAModuleTypeScanController`.
/// @param customClass The custom class which is  inherit from a specific class.
/// @param moduleType The page type corresponding to the class you want to customize the UI. See `AAIIQAModuleType` for more detail.
- (void)registerClass:(Class)customClass forModule:(AAIIQAModuleType)moduleType;

- (Class)classOfModuleType:(AAIIQAModuleType)moduleType;

/// This method is only available for the flutter plugin.
- (void)setPropertyValue:(id)value forPropertyKey:(NSString *)key;

@end

NS_ASSUME_NONNULL_END
